import torch
import torch.nn as nn
from spikingjelly.clock_driven.neuron import MultiStepLIFNode
from timm.models.layers import to_2tuple, trunc_normal_, DropPath
from timm.models import register_model
from timm.models.vision_transformer import _cfg
import torch.nn.functional as F
from thop import profile
from timm.models import create_model

from functools import partial
__all__ = ['spikformer']


class MLP(nn.Module):
    def __init__(self, in_features, hidden_features=None, out_features=None, drop=0.):
        super().__init__()
        out_features = out_features or in_features
        hidden_features = hidden_features = in_features // 3 * 16
        self.fc1_linear = nn.Linear(in_features, hidden_features)
        self.fc1_bn = nn.BatchNorm1d(hidden_features)
        self.fc1_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.dw_conv = nn.Conv2d(hidden_features // 2, hidden_features // 2, kernel_size=3, stride=1, padding=1, groups=hidden_features // 2, bias=False)
        self.dw_bn = nn.BatchNorm2d(hidden_features // 2)
        self.dw_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.fc2_linear = nn.Linear(hidden_features // 2, out_features)
        self.fc2_bn = nn.BatchNorm1d(out_features)
        self.fc2_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.c_hidden = hidden_features
        self.c_output = out_features

    def forward(self, x):
        T,B,N,C = x.shape
        H, W = 8, 8
        x_ = x.flatten(0, 1)
        x = self.fc1_linear(x_)
        x = self.fc1_bn(x.transpose(-1, -2)).transpose(-1, -2).reshape(T, B, N, self.c_hidden).contiguous()
        x = self.fc1_lif(x)
        x1, x2 = torch.chunk(x, 2, dim=3)
        x1 = self.dw_conv(x1.reshape(T*B, H, W, self.c_hidden // 2).permute(0, 3, 1, 2).contiguous())
        x1 = self.dw_bn(x1)
        x1 = self.dw_lif(x1.reshape(T, B, self.c_hidden // 2, H, W).permute(0, 1, 3, 4, 2).contiguous()).reshape(T, B, N, self.c_hidden // 2)
        x = x1 * x2
        x = self.fc2_linear(x.flatten(0,1))
        x = self.fc2_bn(x.transpose(-1, -2)).transpose(-1, -2).reshape(T, B, N, C).contiguous()
        x = self.fc2_lif(x)
        return x


class MLP_Unit(nn.Module):
    def __init__(self, in_features, out_features, drop=0.0, expert_idx=-1):
        super().__init__()
        self.unit_linear = nn.Linear(in_features, out_features)
        self.unit_bn = nn.BatchNorm1d(out_features)
        self.unit_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.out_features = out_features
        self.expert_idx = expert_idx

    def forward(self, x, hook=None):
        TB, N, C = x.shape
        x = self.unit_linear(x)
        x = self.unit_bn(x.transpose(-1, -2)).transpose(-1, -2).reshape(4, TB//4, N, self.out_features).contiguous()
        x = self.unit_lif(x)
        return x


class SSA(nn.Module):
    def __init__(self, dim, expert_dim=0, qkv_bias=False, qk_scale=None, attn_drop=0., proj_drop=0., sr_ratio=1,
                 num_expert=0, top_k=0, gate='complete', attn_scale=0., mode='small'):
        super().__init__()
        self.dim = dim
        self.scale_1 = 0.125
        if mode == 'base':
            self.d = dim
        elif mode == 'small':
            self.d = expert_dim

        self.k_linear = nn.Linear(dim, expert_dim)
        self.k_bn = nn.BatchNorm1d(expert_dim)
        self.k_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.v_linear = nn.Linear(dim, self.d)
        self.v_bn = nn.BatchNorm1d(self.d)
        self.v_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.attn_lif = MultiStepLIFNode(tau=2.0, v_threshold=0.5, detach_reset=True, backend='cupy')

        self.proj_linear = nn.Linear(self.d, dim)
        self.proj_bn = nn.BatchNorm1d(dim)
        self.proj_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.num_expert = num_expert
        self.expert_dim = expert_dim

        self.router1 = nn.Linear(dim, num_expert)
        self.router2 = nn.BatchNorm1d(num_expert)
        self.router3 = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.ff_list = nn.ModuleList([MLP_Unit(dim, expert_dim, 0.0, i) for i in range(num_expert)])
        self.lif_list = nn.ModuleList([MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy') for i in range(num_expert)])

        scale = torch.Tensor([0.1])
        self.scale = nn.Parameter(scale, requires_grad=True)

    def forward(self, x):
        T,B,N,C = x.shape
        x_for_qkv = x.flatten(0, 1)
        k_linear_out = self.k_linear(x_for_qkv)
        k_linear_out = self.k_bn(k_linear_out.transpose(-1, -2)).transpose(-1, -2).reshape(T, B, N, self.expert_dim).contiguous()
        k = self.k_lif(k_linear_out)

        v_linear_out = self.v_linear(x_for_qkv)
        v_linear_out = self.v_bn(v_linear_out.transpose(-1, -2)).transpose(-1, -2).reshape(T, B, N, -1).contiguous()
        v = self.v_lif(v_linear_out)

        weights = self.router1(x_for_qkv)
        weights = self.router2(weights.transpose(-1, -2)).reshape(T, B, N, self.num_expert).contiguous()
        weights = self.router3(weights)  # * self.scale

        y = 0
        for idx in range(self.num_expert):
            weight_idx = weights[:, :, :, idx].unsqueeze(dim=-1)
            q = self.ff_list[idx](x_for_qkv)
            attn = q @ k.transpose(-1, -2)
            result = (attn @ v) * self.scale_1
            result = self.lif_list[idx](result)
            y += weight_idx * result
        y = y.flatten(0, 1)
        y = self.proj_lif(self.proj_bn(self.proj_linear(y).transpose(-1, -2)).transpose(-1, -2).reshape(T, B, N, C))
        return y


class Block(nn.Module):
    def __init__(self, dim, expert_dim=0, mlp_ratio=4., qkv_bias=False, qk_scale=None, drop=0., attn_drop=0.,
                 drop_path=0., norm_layer=nn.LayerNorm, sr_ratio=1, num_expert=0,):
        super().__init__()
        self.norm1 = norm_layer(dim)
        self.attn = SSA(dim, expert_dim=expert_dim, qkv_bias=qkv_bias, qk_scale=qk_scale,
                              attn_drop=attn_drop, proj_drop=drop, sr_ratio=sr_ratio, num_expert=4,mode='base')
        self.norm2 = norm_layer(dim)
        mlp_hidden_dim = int(dim * mlp_ratio)
        self.mlp = MLP(in_features=dim, hidden_features=mlp_hidden_dim, drop=drop)

    def forward(self, x):
        x = x + self.attn(x)
        x = x + self.mlp(x)
        return x


class SPS(nn.Module):
    def __init__(self, img_size_h=128, img_size_w=128, patch_size=4, in_channels=2, embed_dims=256):
        super().__init__()
        self.image_size = [img_size_h, img_size_w]
        patch_size = to_2tuple(patch_size)
        self.patch_size = patch_size
        self.C = in_channels
        self.H, self.W = self.image_size[0] // patch_size[0], self.image_size[1] // patch_size[1]
        self.num_patches = self.H * self.W
        self.proj_conv = nn.Conv2d(in_channels, embed_dims//8, kernel_size=3, stride=1, padding=1, bias=False)
        self.proj_bn = nn.BatchNorm2d(embed_dims//8)
        self.proj_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.proj_conv1 = nn.Conv2d(embed_dims//8, embed_dims//4, kernel_size=3, stride=1, padding=1, bias=False)
        self.proj_bn1 = nn.BatchNorm2d(embed_dims//4)
        self.proj_lif1 = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

        self.proj_conv2 = nn.Conv2d(embed_dims//4, embed_dims//2, kernel_size=3, stride=1, padding=1, bias=False)
        self.proj_bn2 = nn.BatchNorm2d(embed_dims//2)
        self.proj_lif2 = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')
        self.maxpool2 = torch.nn.MaxPool2d(kernel_size=3, stride=2, padding=1, dilation=1, ceil_mode=False)

        self.proj_conv3 = nn.Conv2d(embed_dims//2, embed_dims, kernel_size=3, stride=1, padding=1, bias=False)
        self.proj_bn3 = nn.BatchNorm2d(embed_dims)
        self.proj_lif3 = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')
        self.maxpool3 = torch.nn.MaxPool2d(kernel_size=3, stride=2, padding=1, dilation=1, ceil_mode=False)

        self.rpe_conv = nn.Conv2d(embed_dims, embed_dims, kernel_size=3, stride=1, padding=1, bias=False)
        self.rpe_bn = nn.BatchNorm2d(embed_dims)
        self.rpe_lif = MultiStepLIFNode(tau=2.0, detach_reset=True, backend='cupy')

    def forward(self, x):
        T, B, C, H, W = x.shape
        x = self.proj_conv(x.flatten(0, 1)) # have some fire value
        x = self.proj_bn(x).reshape(T, B, -1, H, W).contiguous()
        x = self.proj_lif(x).flatten(0, 1).contiguous()

        x = self.proj_conv1(x)
        x = self.proj_bn1(x).reshape(T, B, -1, H, W).contiguous()
        x = self.proj_lif1(x).flatten(0, 1).contiguous()

        x = self.proj_conv2(x)
        x = self.proj_bn2(x).reshape(T, B, -1, H, W).contiguous()
        x = self.proj_lif2(x).flatten(0, 1).contiguous()
        x = self.maxpool2(x)

        x = self.proj_conv3(x)
        x = self.proj_bn3(x).reshape(T, B, -1, H//2, W//2).contiguous()
        x = self.proj_lif3(x).flatten(0, 1).contiguous()
        x = self.maxpool3(x)

        x_feat = x.reshape(T, B, -1, H//4, W//4).contiguous()
        x = self.rpe_conv(x)
        x = self.rpe_bn(x).reshape(T, B, -1, H//4, W//4).contiguous()
        x = self.rpe_lif(x)
        x = x + x_feat

        x = x.flatten(-2).transpose(-1, -2)  # T,B,N,C
        return x


class Spikformer(nn.Module):
    def __init__(self,
                 img_size_h=128, img_size_w=128, patch_size=16, in_channels=2, num_classes=11,
                 embed_dims=[64, 128, 256], expert_dim=0, mlp_ratios=[4, 4, 4], qkv_bias=False, qk_scale=None,
                 drop_rate=0., attn_drop_rate=0., drop_path_rate=0., norm_layer=nn.LayerNorm,
                 depths=[6, 8, 6], sr_ratios=[8, 4, 2], T = 4, num_expert=0, top_k=0,
                 gate='complete'
                 ):
        super().__init__()
        self.T = T  # time step
        self.num_classes = num_classes
        self.depths = depths

        dpr = [x.item() for x in torch.linspace(0, drop_path_rate, depths)]  # stochastic depth decay rule
        attn_scale = [0.25, 0.5, 0.75, 1]
        block_scale = [1/3, 1/3, 1/3, 1/3]

        patch_embed = SPS(img_size_h=img_size_h,
                                 img_size_w=img_size_w,
                                 patch_size=patch_size,
                                 in_channels=in_channels,
                                 embed_dims=embed_dims)

        block = nn.ModuleList([Block(
            dim=embed_dims, expert_dim=expert_dim,  mlp_ratio=mlp_ratios, qkv_bias=qkv_bias,
            qk_scale=qk_scale, drop=drop_rate, attn_drop=attn_drop_rate, drop_path=dpr[j],
            norm_layer=norm_layer, sr_ratio=sr_ratios, num_expert=num_expert)
            for j in range(depths)])
        setattr(self, f"patch_embed", patch_embed)
        setattr(self, f"block", block)

        # classification head
        self.head = nn.Linear(embed_dims, num_classes) if num_classes > 0 else nn.Identity()
        self.apply(self._init_weights)

    @torch.jit.ignore
    def _get_pos_embed(self, pos_embed, patch_embed, H, W):
        if H * W == self.patch_embed1.num_patches:
            return pos_embed
        else:
            return F.interpolate(
                pos_embed.reshape(1, patch_embed.H, patch_embed.W, -1).permute(0, 3, 1, 2),
                size=(H, W), mode="bilinear").reshape(1, -1, H * W).permute(0, 2, 1)

    def _init_weights(self, m):
        if isinstance(m, nn.Linear):
            trunc_normal_(m.weight, std=.02)
            if isinstance(m, nn.Linear) and m.bias is not None:
                nn.init.constant_(m.bias, 0)
        elif isinstance(m, nn.LayerNorm):
            nn.init.constant_(m.bias, 0)
            nn.init.constant_(m.weight, 1.0)

    def forward_features(self, x):

        block = getattr(self, f"block")
        patch_embed = getattr(self, f"patch_embed")

        x = patch_embed(x)
        for blk in block:
            x = blk(x)
        return x.mean(2)

    def forward(self, x):
        x = (x.unsqueeze(0)).repeat(self.T, 1, 1, 1, 1)
        x = self.forward_features(x)
        x = self.head(x.mean(0))
        return x


@register_model
def spikformer(pretrained=False, **kwargs):
    model = Spikformer(
        # img_size_h=224, img_size_w=224,
        # patch_size=16, embed_dims=768, num_heads=12, mlp_ratios=4,
        # in_channels=3, num_classes=1000, qkv_bias=False,
        # norm_layer=partial(nn.LayerNorm, eps=1e-6), depths=12, sr_ratios=1,
        **kwargs
    )
    model.default_cfg = _cfg()
    return model


if __name__ == '__main__':
    # device = torch.device("cpu")

    model = create_model(
        'spikformer',
        pretrained=False,
        drop_rate=0,
        drop_path_rate=0.1,
        drop_block_rate=None,
        img_size_h=32, img_size_w=32,
        patch_size=4, embed_dims=384, expert_dim=96, mlp_ratios=4,
        in_channels=3, num_classes=100, qkv_bias=False,
        norm_layer=partial(nn.LayerNorm, eps=1e-6), depths=4, sr_ratios=1,
        T=4, num_expert=4,
    ).cuda()

    input = torch.ones(2, 3, 32, 32).cuda()
    output = model(input)

    '''
    input = torch.randn(2, 3, 32, 32).cuda()
    flops, params = profile(model, inputs=(input,))
    print("FLOPs=", str(flops / 1e9) + '{}'.format("G"))
    print("Params=", str(params / 1e6) + '{}'.format("M"))
    # output = model(input)
    '''
    '''
    # print(output.shape)
    output = model(input)
    for name, param in model.named_parameters():
        if param.grad is not None:
            print(f"{name}的梯度矩阵：")
            print(param.grad)
    '''
